import React, { useState } from "react";
import axios from "axios";

export default function MovieCard({ movie }) {
  const [isAdding, setIsAdding] = useState(false);
  const [isFavorited, setIsFavorited] = useState(false);

  const addToFavorites = async () => {
    if (isAdding || isFavorited) return;
    
    setIsAdding(true);
    try {
      await axios.post("http://localhost:5000/api/movies/favorites", {
        imdbID: movie.imdbID,
        title: movie.Title,
        poster: movie.Poster,
        year: movie.Year
      });
      setIsFavorited(true);
      setTimeout(() => setIsAdding(false), 500);
    } catch (err) {
      console.error(err);
      alert("Failed to add favorite.");
      setIsAdding(false);
    }
  };

  return (
    <div className="group relative bg-white bg-opacity-10 backdrop-blur-lg rounded-2xl overflow-hidden border border-white border-opacity-20 shadow-xl hover:shadow-2xl transition-all duration-300 transform hover:scale-105">
      {/* Movie Poster */}
      <div className="relative overflow-hidden">
        <img
          src={
            movie.Poster !== "N/A"
              ? movie.Poster
              : "https://via.placeholder.com/300x450?text=No+Image"
          }
          alt={movie.Title}
          className="w-full h-80 object-cover transition-transform duration-500 group-hover:scale-110"
        />
        
        {/* Gradient Overlay */}
        <div className="absolute inset-0 bg-gradient-to-t from-black via-transparent to-transparent opacity-60"></div>
        
        {/* Year Badge */}
        {movie.Year && (
          <div className="absolute top-3 right-3 bg-black bg-opacity-70 text-white px-3 py-1 rounded-full text-sm font-semibold backdrop-blur-sm">
            {movie.Year}
          </div>
        )}

        {/* Quick Add Button (Appears on Hover) */}
        <button
          onClick={addToFavorites}
          disabled={isAdding || isFavorited}
          className="absolute top-3 left-3 bg-rose-500 text-white p-3 rounded-full opacity-0 group-hover:opacity-100 transition-all duration-300 hover:bg-rose-600 hover:scale-110 disabled:opacity-50 disabled:cursor-not-allowed shadow-lg"
          title={isFavorited ? "Already in favorites" : "Add to favorites"}
        >
          {isFavorited ? (
            <svg
              xmlns="http://www.w3.org/2000/svg"
              className="h-5 w-5"
              viewBox="0 0 20 20"
              fill="currentColor"
            >
              <path
                fillRule="evenodd"
                d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                clipRule="evenodd"
              />
            </svg>
          ) : (
            <svg
              xmlns="http://www.w3.org/2000/svg"
              className="h-5 w-5"
              viewBox="0 0 20 20"
              fill="currentColor"
            >
              <path
                fillRule="evenodd"
                d="M3.172 5.172a4 4 0 015.656 0L10 6.343l1.172-1.171a4 4 0 115.656 5.656L10 17.657l-6.828-6.829a4 4 0 010-5.656z"
                clipRule="evenodd"
              />
            </svg>
          )}
        </button>
      </div>

      {/* Movie Info */}
      <div className="p-4">
        <h3 className="text-gray-900 font-semibold text-lg line-clamp-2 mb-3 min-h-[3.5rem]">
          {movie.Title}
        </h3>

        {/* Add to Favorites Button */}
        <button
          onClick={addToFavorites}
          disabled={isAdding || isFavorited}
          className={`w-full py-3 px-4 rounded-xl font-semibold transition-all duration-300 transform hover:scale-105 disabled:cursor-not-allowed shadow-lg flex items-center justify-center gap-2 ${
            isFavorited
              ? "bg-green-500 text-white"
              : "bg-gradient-to-r from-rose-500 to-pink-600 text-white hover:from-rose-600 hover:to-pink-700"
          }`}
        >
          {isAdding ? (
            <>
              <svg
                className="animate-spin h-5 w-5"
                xmlns="http://www.w3.org/2000/svg"
                fill="none"
                viewBox="0 0 24 24"
              >
                <circle
                  className="opacity-25"
                  cx="12"
                  cy="12"
                  r="10"
                  stroke="currentColor"
                  strokeWidth="4"
                ></circle>
                <path
                  className="opacity-75"
                  fill="currentColor"
                  d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                ></path>
              </svg>
              Adding...
            </>
          ) : isFavorited ? (
            <>
              <svg
                xmlns="http://www.w3.org/2000/svg"
                className="h-5 w-5"
                viewBox="0 0 20 20"
                fill="currentColor"
              >
                <path
                  fillRule="evenodd"
                  d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                  clipRule="evenodd"
                />
              </svg>
              Added
            </>
          ) : (
            <>
              Add to Favorites
            </>
          )}
        </button>
      </div>

      {/* Type Badge (if available) */}
      {movie.Type && (
        <div className="absolute bottom-20 right-3 bg-purple-600 bg-opacity-90 text-white px-3 py-1 rounded-full text-xs font-semibold uppercase backdrop-blur-sm">
          {movie.Type}
        </div>
      )}
    </div>
  );
}